<?php
/**
 * الصفحة الرئيسية - لوحة التحكم
 * Main Dashboard - Live Streaming Control Panel
 */

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/db.php';

// التحقق من تسجيل الدخول
if (!isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// تنظيف الرسائل القديمة
cleanupOldMessages();

// الحصول على الأجهزة المسجلة
$devices = getAllDevices();
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة التحكم - البث المباشر</title>
    <link href="https://fonts.googleapis.com/css2?family=Tajawal:wght@400;500;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary: #6366f1;
            --primary-dark: #4f46e5;
            --secondary: #ec4899;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --dark: #1e1b4b;
            --darker: #0f0d24;
            --light: #f8fafc;
            --gray: #64748b;
            --card-bg: rgba(30, 27, 75, 0.8);
            --glass: rgba(255, 255, 255, 0.05);
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Tajawal', sans-serif;
            background: linear-gradient(135deg, var(--darker) 0%, var(--dark) 50%, #1a1a2e 100%);
            min-height: 100vh;
            color: var(--light);
        }
        
        .navbar {
            background: var(--glass);
            backdrop-filter: blur(20px);
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            position: sticky;
            top: 0;
            z-index: 1000;
        }
        
        .navbar h1 {
            font-size: 1.5rem;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .navbar-links {
            display: flex;
            gap: 1rem;
            align-items: center;
        }
        
        .navbar-links a {
            color: var(--light);
            text-decoration: none;
            padding: 0.5rem 1rem;
            border-radius: 8px;
            transition: all 0.3s;
        }
        
        .navbar-links a:hover {
            background: var(--glass);
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .stat-card {
            background: var(--card-bg);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 1.5rem;
            border: 1px solid rgba(255,255,255,0.1);
            transition: all 0.3s;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            border-color: var(--primary);
        }
        
        .stat-card .icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .stat-card.devices .icon { background: linear-gradient(135deg, var(--primary), var(--secondary)); }
        .stat-card.online .icon { background: linear-gradient(135deg, var(--success), #059669); }
        .stat-card.streams .icon { background: linear-gradient(135deg, var(--warning), #d97706); }
        
        .stat-card h3 {
            color: var(--gray);
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
        }
        
        .stat-card .value {
            font-size: 2rem;
            font-weight: 700;
        }
        
        .section-title {
            font-size: 1.5rem;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .devices-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
            gap: 1.5rem;
        }
        
        .device-card {
            background: var(--card-bg);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 1.5rem;
            border: 1px solid rgba(255,255,255,0.1);
            transition: all 0.3s;
        }
        
        .device-card:hover {
            border-color: var(--primary);
        }
        
        .device-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 1rem;
        }
        
        .device-info h3 {
            font-size: 1.2rem;
            margin-bottom: 0.25rem;
        }
        
        .device-info p {
            color: var(--gray);
            font-size: 0.85rem;
        }
        
        .device-status {
            padding: 0.25rem 0.75rem;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 500;
        }
        
        .device-status.online {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success);
        }
        
        .device-status.offline {
            background: rgba(239, 68, 68, 0.2);
            color: var(--danger);
        }
        
        .device-meta {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            margin-bottom: 1rem;
            padding: 1rem;
            background: var(--glass);
            border-radius: 12px;
        }
        
        .device-meta-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.85rem;
            color: var(--gray);
        }
        
        .device-meta-item i {
            color: var(--primary);
        }
        
        .device-actions {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 0.75rem;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            padding: 0.75rem 1rem;
            border-radius: 10px;
            font-family: inherit;
            font-size: 0.9rem;
            font-weight: 500;
            cursor: pointer;
            border: none;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(99, 102, 241, 0.4);
        }
        
        .btn-secondary {
            background: var(--glass);
            color: var(--light);
            border: 1px solid rgba(255,255,255,0.2);
        }
        
        .btn-secondary:hover {
            background: rgba(255,255,255,0.1);
        }
        
        .btn-success {
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
        }
        
        .btn-danger {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .btn:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            background: var(--card-bg);
            border-radius: 16px;
            border: 2px dashed rgba(255,255,255,0.1);
        }
        
        .empty-state i {
            font-size: 4rem;
            color: var(--gray);
            margin-bottom: 1rem;
        }
        
        .empty-state h3 {
            margin-bottom: 0.5rem;
        }
        
        .empty-state p {
            color: var(--gray);
        }
        
        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.8);
            backdrop-filter: blur(5px);
            z-index: 2000;
            align-items: center;
            justify-content: center;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: var(--dark);
            border-radius: 20px;
            width: 90%;
            max-width: 900px;
            max-height: 90vh;
            overflow: hidden;
            border: 1px solid rgba(255,255,255,0.1);
        }
        
        .modal-header {
            padding: 1.5rem;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h2 {
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .modal-close {
            background: none;
            border: none;
            color: var(--gray);
            font-size: 1.5rem;
            cursor: pointer;
            transition: color 0.3s;
        }
        
        .modal-close:hover {
            color: var(--danger);
        }
        
        .modal-body {
            padding: 1.5rem;
        }
        
        .stream-options {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }
        
        .stream-option {
            background: var(--glass);
            border: 2px solid rgba(255,255,255,0.1);
            border-radius: 12px;
            padding: 1.5rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .stream-option:hover {
            border-color: var(--primary);
        }
        
        .stream-option.selected {
            border-color: var(--primary);
            background: rgba(99, 102, 241, 0.1);
        }
        
        .stream-option i {
            font-size: 2.5rem;
            margin-bottom: 1rem;
            color: var(--primary);
        }
        
        .stream-option h4 {
            margin-bottom: 0.5rem;
        }
        
        .stream-option p {
            font-size: 0.85rem;
            color: var(--gray);
        }
        
        .video-container {
            background: #000;
            border-radius: 12px;
            overflow: hidden;
            aspect-ratio: 16/9;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 1rem;
        }
        
        .video-container video {
            width: 100%;
            height: 100%;
            object-fit: contain;
        }
        
        .video-placeholder {
            text-align: center;
            color: var(--gray);
        }
        
        .video-placeholder i {
            font-size: 4rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }
        
        .stream-controls {
            display: flex;
            justify-content: center;
            gap: 1rem;
        }
        
        .stream-status {
            text-align: center;
            padding: 1rem;
            background: var(--glass);
            border-radius: 10px;
            margin-bottom: 1rem;
        }
        
        .stream-status.connecting {
            color: var(--warning);
        }
        
        .stream-status.active {
            color: var(--success);
        }
        
        .stream-status.error {
            color: var(--danger);
        }
        
        /* تحديث تلقائي للأجهزة */
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .updating {
            animation: pulse 1s infinite;
        }
        
        /* Responsive */
        @media (max-width: 768px) {
            .container {
                padding: 1rem;
            }
            
            .devices-grid {
                grid-template-columns: 1fr;
            }
            
            .device-actions {
                grid-template-columns: 1fr;
            }
            
            .modal-content {
                width: 95%;
                max-height: 95vh;
            }
        }
    </style>
</head>
<body>
    <nav class="navbar">
        <h1><i class="fas fa-broadcast-tower"></i> لوحة التحكم</h1>
        <div class="navbar-links">
            <a href="index.php"><i class="fas fa-home"></i> الرئيسية</a>
            <a href="logs.php"><i class="fas fa-list"></i> السجلات</a>
            <a href="logout.php"><i class="fas fa-sign-out-alt"></i> خروج</a>
        </div>
    </nav>
    
    <div class="container">
        <!-- الإحصائيات -->
        <div class="stats-grid">
            <div class="stat-card devices">
                <div class="icon"><i class="fas fa-mobile-alt"></i></div>
                <h3>إجمالي الأجهزة</h3>
                <div class="value"><?php echo count($devices); ?></div>
            </div>
            <div class="stat-card online">
                <div class="icon"><i class="fas fa-wifi"></i></div>
                <h3>الأجهزة المتصلة</h3>
                <div class="value" id="online-count"><?php echo count(array_filter($devices, fn($d) => $d['is_online'])); ?></div>
            </div>
            <div class="stat-card streams">
                <div class="icon"><i class="fas fa-video"></i></div>
                <h3>البث النشط</h3>
                <div class="value" id="active-streams">0</div>
            </div>
        </div>
        
        <!-- قائمة الأجهزة -->
        <h2 class="section-title"><i class="fas fa-mobile-alt"></i> الأجهزة المسجلة</h2>
        
        <?php if (empty($devices)): ?>
        <div class="empty-state">
            <i class="fas fa-mobile-alt"></i>
            <h3>لا توجد أجهزة مسجلة</h3>
            <p>عند تثبيت التطبيق على جهاز، سيظهر هنا تلقائياً</p>
        </div>
        <?php else: ?>
        <div class="devices-grid" id="devices-container">
            <?php foreach ($devices as $device): ?>
            <div class="device-card" data-device-id="<?php echo htmlspecialchars($device['device_id']); ?>">
                <div class="device-header">
                    <div class="device-info">
                        <h3><?php echo htmlspecialchars($device['device_name'] ?: 'جهاز غير مسمى'); ?></h3>
                        <p><?php echo htmlspecialchars($device['device_model']); ?></p>
                    </div>
                    <span class="device-status <?php echo $device['is_online'] ? 'online' : 'offline'; ?>">
                        <?php echo $device['is_online'] ? 'متصل' : 'غير متصل'; ?>
                    </span>
                </div>
                
                <div class="device-meta">
                    <div class="device-meta-item">
                        <i class="fab fa-android"></i>
                        <span>Android <?php echo htmlspecialchars($device['android_version'] ?: 'N/A'); ?></span>
                    </div>
                    <div class="device-meta-item">
                        <i class="fas fa-clock"></i>
                        <span><?php echo $device['last_seen'] ? date('Y-m-d H:i', strtotime($device['last_seen'])) : 'N/A'; ?></span>
                    </div>
                    <div class="device-meta-item">
                        <i class="fas fa-fingerprint"></i>
                        <span><?php echo substr($device['device_id'], 0, 8); ?>...</span>
                    </div>
                </div>
                
                <div class="device-actions">
                    <button class="btn btn-primary" onclick="openStreamModal('<?php echo $device['device_id']; ?>', 'camera')" 
                            <?php echo !$device['is_online'] ? 'disabled' : ''; ?>>
                        <i class="fas fa-video"></i> كاميرا مباشر
                    </button>
                    <button class="btn btn-secondary" onclick="openStreamModal('<?php echo $device['device_id']; ?>', 'screen')"
                            <?php echo !$device['is_online'] ? 'disabled' : ''; ?>>
                        <i class="fas fa-desktop"></i> شاشة مباشر
                    </button>
                    <button class="btn btn-success" onclick="openStreamModal('<?php echo $device['device_id']; ?>', 'audio')"
                            <?php echo !$device['is_online'] ? 'disabled' : ''; ?>>
                        <i class="fas fa-microphone"></i> صوت مباشر
                    </button>
                    <button class="btn btn-secondary" onclick="viewDeviceLogs('<?php echo $device['device_id']; ?>')">
                        <i class="fas fa-history"></i> السجلات
                    </button>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Modal للبث -->
    <div class="modal" id="streamModal">
        <div class="modal-content">
            <div class="modal-header">
                <h2><i class="fas fa-broadcast-tower"></i> <span id="modalTitle">البث المباشر</span></h2>
                <button class="modal-close" onclick="closeStreamModal()"><i class="fas fa-times"></i></button>
            </div>
            <div class="modal-body">
                <!-- خيارات البث -->
                <div id="streamOptions">
                    <div class="stream-options" id="cameraOptions" style="display: none;">
                        <div class="stream-option selected" data-camera="front" onclick="selectCamera('front')">
                            <i class="fas fa-camera"></i>
                            <h4>الكاميرا الأمامية</h4>
                            <p>للمكالمات والسيلفي</p>
                        </div>
                        <div class="stream-option" data-camera="back" onclick="selectCamera('back')">
                            <i class="fas fa-camera-retro"></i>
                            <h4>الكاميرا الخلفية</h4>
                            <p>للتصوير العادي</p>
                        </div>
                    </div>
                    
                    <div class="stream-controls">
                        <button class="btn btn-primary" onclick="startStream()">
                            <i class="fas fa-play"></i> بدء البث
                        </button>
                    </div>
                </div>
                
                <!-- عرض البث -->
                <div id="streamViewer" style="display: none;">
                    <div class="stream-status connecting" id="streamStatus">
                        <i class="fas fa-spinner fa-spin"></i> جاري الاتصال...
                    </div>
                    
                    <div class="video-container">
                        <video id="remoteVideo" autoplay playsinline></video>
                        <div class="video-placeholder" id="videoPlaceholder">
                            <i class="fas fa-video-slash"></i>
                            <p>في انتظار البث...</p>
                        </div>
                    </div>
                    
                    <div class="stream-controls">
                        <button class="btn btn-danger" onclick="stopStream()">
                            <i class="fas fa-stop"></i> إيقاف البث
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // متغيرات عامة
        let currentDeviceId = null;
        let currentStreamType = null;
        let currentCamera = 'front';
        let currentSessionId = null;
        let peerConnection = null;
        let pollingInterval = null;
        
        // إعدادات ICE
        const iceServers = <?php echo ICE_SERVERS; ?>;
        
        // فتح Modal البث
        function openStreamModal(deviceId, streamType) {
            currentDeviceId = deviceId;
            currentStreamType = streamType;
            
            const modal = document.getElementById('streamModal');
            const modalTitle = document.getElementById('modalTitle');
            const cameraOptions = document.getElementById('cameraOptions');
            const streamOptions = document.getElementById('streamOptions');
            const streamViewer = document.getElementById('streamViewer');
            
            // إعادة تعيين
            streamOptions.style.display = 'block';
            streamViewer.style.display = 'none';
            
            // تعيين العنوان وإظهار الخيارات حسب النوع
            switch (streamType) {
                case 'camera':
                    modalTitle.textContent = 'بث الكاميرا المباشر';
                    cameraOptions.style.display = 'grid';
                    break;
                case 'screen':
                    modalTitle.textContent = 'بث الشاشة المباشر';
                    cameraOptions.style.display = 'none';
                    break;
                case 'audio':
                    modalTitle.textContent = 'بث الصوت المباشر';
                    cameraOptions.style.display = 'none';
                    break;
            }
            
            modal.classList.add('active');
        }
        
        // إغلاق Modal
        function closeStreamModal() {
            stopStream();
            document.getElementById('streamModal').classList.remove('active');
            currentDeviceId = null;
            currentStreamType = null;
            currentSessionId = null;
        }
        
        // اختيار الكاميرا
        function selectCamera(camera) {
            currentCamera = camera;
            document.querySelectorAll('.stream-option').forEach(opt => {
                opt.classList.remove('selected');
            });
            document.querySelector(`[data-camera="${camera}"]`).classList.add('selected');
        }
        
        // بدء البث
        async function startStream() {
            try {
                // إنشاء جلسة
                const response = await fetch('api/stream.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'create_session',
                        device_id: currentDeviceId,
                        stream_type: currentStreamType,
                        camera_facing: currentStreamType === 'camera' ? currentCamera : null
                    })
                });
                
                const data = await response.json();
                
                if (!data.success) {
                    throw new Error(data.error || 'فشل إنشاء الجلسة');
                }
                
                currentSessionId = data.session_id;
                
                // إظهار عارض البث
                document.getElementById('streamOptions').style.display = 'none';
                document.getElementById('streamViewer').style.display = 'block';
                
                // إعداد WebRTC
                await setupWebRTC();
                
                // بدء polling للرسائل
                startPolling();
                
            } catch (error) {
                console.error('Error starting stream:', error);
                updateStreamStatus('error', 'خطأ: ' + error.message);
            }
        }
        
        // إعداد WebRTC
        async function setupWebRTC() {
            const configuration = { iceServers: iceServers };
            peerConnection = new RTCPeerConnection(configuration);
            
            // عند استلام track
            peerConnection.ontrack = (event) => {
                console.log('Received remote track:', event.track.kind);
                const video = document.getElementById('remoteVideo');
                const placeholder = document.getElementById('videoPlaceholder');
                
                if (event.streams && event.streams[0]) {
                    video.srcObject = event.streams[0];
                    placeholder.style.display = 'none';
                    video.style.display = 'block';
                    updateStreamStatus('active', 'البث نشط');
                }
            };
            
            // عند تغيير حالة الاتصال
            peerConnection.onconnectionstatechange = () => {
                console.log('Connection state:', peerConnection.connectionState);
                switch (peerConnection.connectionState) {
                    case 'connected':
                        updateStreamStatus('active', 'متصل');
                        break;
                    case 'disconnected':
                    case 'failed':
                        updateStreamStatus('error', 'انقطع الاتصال');
                        break;
                }
            };
            
            // عند الحصول على ICE candidate
            peerConnection.onicecandidate = async (event) => {
                if (event.candidate) {
                    await sendSignalingMessage('ice_candidate', event.candidate);
                }
            };
            
            // إنشاء offer
            const offer = await peerConnection.createOffer({
                offerToReceiveVideo: currentStreamType !== 'audio',
                offerToReceiveAudio: currentStreamType !== 'screen'
            });
            
            await peerConnection.setLocalDescription(offer);
            await sendSignalingMessage('offer', offer);
            
            updateStreamStatus('connecting', 'جاري الاتصال بالجهاز...');
        }
        
        // إرسال رسالة signaling
        async function sendSignalingMessage(type, payload) {
            try {
                await fetch('api/signaling.php', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        action: 'send',
                        session_id: currentSessionId,
                        message_type: type,
                        payload: JSON.stringify(payload)
                    })
                });
            } catch (error) {
                console.error('Error sending signaling message:', error);
            }
        }
        
        // بدء polling للرسائل
        function startPolling() {
            pollingInterval = setInterval(async () => {
                try {
                    const response = await fetch(`api/signaling.php?action=receive&session_id=${currentSessionId}&for_type=panel`);
                    const data = await response.json();
                    
                    if (data.messages && data.messages.length > 0) {
                        for (const msg of data.messages) {
                            await handleSignalingMessage(msg);
                        }
                    }
                } catch (error) {
                    console.error('Polling error:', error);
                }
            }, 1000);
        }
        
        // معالجة رسائل signaling
        async function handleSignalingMessage(msg) {
            const payload = JSON.parse(msg.payload);
            
            switch (msg.message_type) {
                case 'answer':
                    await peerConnection.setRemoteDescription(new RTCSessionDescription(payload));
                    break;
                    
                case 'ice_candidate':
                    await peerConnection.addIceCandidate(new RTCIceCandidate(payload));
                    break;
                    
                case 'screen_off':
                    updateStreamStatus('error', 'شاشة الهاتف مغلقة');
                    break;
                    
                case 'error':
                    updateStreamStatus('error', payload.message || 'حدث خطأ');
                    break;
            }
        }
        
        // تحديث حالة البث
        function updateStreamStatus(status, message) {
            const statusEl = document.getElementById('streamStatus');
            statusEl.className = 'stream-status ' + status;
            statusEl.innerHTML = `<i class="fas fa-${status === 'connecting' ? 'spinner fa-spin' : status === 'active' ? 'check-circle' : 'exclamation-circle'}"></i> ${message}`;
        }
        
        // إيقاف البث
        async function stopStream() {
            if (pollingInterval) {
                clearInterval(pollingInterval);
                pollingInterval = null;
            }
            
            if (peerConnection) {
                peerConnection.close();
                peerConnection = null;
            }
            
            if (currentSessionId) {
                try {
                    await fetch('api/stream.php', {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({
                            action: 'end_session',
                            session_id: currentSessionId
                        })
                    });
                } catch (error) {
                    console.error('Error ending session:', error);
                }
            }
            
            // إعادة تعيين الفيديو
            const video = document.getElementById('remoteVideo');
            if (video.srcObject) {
                video.srcObject.getTracks().forEach(track => track.stop());
                video.srcObject = null;
            }
            document.getElementById('videoPlaceholder').style.display = 'flex';
        }
        
        // عرض سجلات الجهاز
        function viewDeviceLogs(deviceId) {
            window.location.href = 'logs.php?device_id=' + deviceId;
        }
        
        // تحديث حالة الأجهزة تلقائياً
        setInterval(async () => {
            try {
                const response = await fetch('api/devices.php?action=status');
                const data = await response.json();
                
                if (data.devices) {
                    document.getElementById('online-count').textContent = 
                        data.devices.filter(d => d.is_online).length;
                    
                    data.devices.forEach(device => {
                        const card = document.querySelector(`[data-device-id="${device.device_id}"]`);
                        if (card) {
                            const status = card.querySelector('.device-status');
                            const buttons = card.querySelectorAll('button');
                            
                            status.className = 'device-status ' + (device.is_online ? 'online' : 'offline');
                            status.textContent = device.is_online ? 'متصل' : 'غير متصل';
                            
                            buttons.forEach((btn, index) => {
                                if (index < 3) { // أول 3 أزرار للبث
                                    btn.disabled = !device.is_online;
                                }
                            });
                        }
                    });
                }
            } catch (error) {
                console.error('Error updating device status:', error);
            }
        }, 10000);
    </script>
</body>
</html>
