<?php
/**
 * ملف قاعدة البيانات - SQLite
 * Database Handler using SQLite (No installation required)
 */

require_once __DIR__ . '/config.php';

class Database {
    private static $instance = null;
    private $db;
    
    private function __construct() {
        // إنشاء مجلد البيانات إذا لم يكن موجوداً
        if (!is_dir(DATA_DIR)) {
            mkdir(DATA_DIR, 0755, true);
        }
        
        try {
            $this->db = new PDO('sqlite:' . DB_PATH);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->db->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            $this->initTables();
        } catch (PDOException $e) {
            die('Database Error: ' . $e->getMessage());
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->db;
    }
    
    private function initTables() {
        // جدول الأجهزة المسجلة
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS devices (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                device_id TEXT UNIQUE NOT NULL,
                device_name TEXT,
                device_model TEXT,
                android_version TEXT,
                last_seen DATETIME,
                is_online INTEGER DEFAULT 0,
                ip_address TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // جدول جلسات البث
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS stream_sessions (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                session_id TEXT UNIQUE NOT NULL,
                device_id TEXT NOT NULL,
                stream_type TEXT NOT NULL,
                camera_facing TEXT,
                status TEXT DEFAULT 'pending',
                started_at DATETIME,
                ended_at DATETIME,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (device_id) REFERENCES devices(device_id)
            )
        ");
        
        // جدول السجلات
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS activity_logs (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                device_id TEXT,
                action TEXT,
                details TEXT,
                ip_address TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        // جدول رسائل Signaling
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS signaling_messages (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                session_id TEXT NOT NULL,
                from_type TEXT NOT NULL,
                to_type TEXT NOT NULL,
                message_type TEXT NOT NULL,
                payload TEXT,
                is_read INTEGER DEFAULT 0,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
    }
}

// دوال مساعدة لقاعدة البيانات

function getDB() {
    return Database::getInstance()->getConnection();
}

// تسجيل جهاز جديد
function registerDevice($deviceId, $deviceName, $deviceModel = '', $androidVersion = '') {
    $db = getDB();
    
    $stmt = $db->prepare("
        INSERT INTO devices (device_id, device_name, device_model, android_version, last_seen, is_online, ip_address)
        VALUES (:device_id, :device_name, :device_model, :android_version, datetime('now'), 1, :ip_address)
        ON CONFLICT(device_id) DO UPDATE SET
            device_name = :device_name,
            device_model = :device_model,
            android_version = :android_version,
            last_seen = datetime('now'),
            is_online = 1,
            ip_address = :ip_address
    ");
    
    return $stmt->execute([
        ':device_id' => $deviceId,
        ':device_name' => $deviceName,
        ':device_model' => $deviceModel,
        ':android_version' => $androidVersion,
        ':ip_address' => getClientIP()
    ]);
}

// الحصول على جميع الأجهزة
function getAllDevices() {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM devices ORDER BY last_seen DESC");
    return $stmt->fetchAll();
}

// الحصول على جهاز محدد
function getDevice($deviceId) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM devices WHERE device_id = :device_id");
    $stmt->execute([':device_id' => $deviceId]);
    return $stmt->fetch();
}

// تحديث حالة الجهاز
function updateDeviceStatus($deviceId, $isOnline) {
    $db = getDB();
    $stmt = $db->prepare("
        UPDATE devices 
        SET is_online = :is_online, last_seen = datetime('now')
        WHERE device_id = :device_id
    ");
    return $stmt->execute([
        ':device_id' => $deviceId,
        ':is_online' => $isOnline ? 1 : 0
    ]);
}

// إنشاء جلسة بث جديدة
function createStreamSession($deviceId, $streamType, $cameraFacing = null) {
    $db = getDB();
    $sessionId = generateUniqueId(32);
    
    $stmt = $db->prepare("
        INSERT INTO stream_sessions (session_id, device_id, stream_type, camera_facing, status)
        VALUES (:session_id, :device_id, :stream_type, :camera_facing, 'pending')
    ");
    
    $stmt->execute([
        ':session_id' => $sessionId,
        ':device_id' => $deviceId,
        ':stream_type' => $streamType,
        ':camera_facing' => $cameraFacing
    ]);
    
    return $sessionId;
}

// تحديث حالة جلسة البث
function updateStreamSession($sessionId, $status) {
    $db = getDB();
    
    $updates = ['status' => $status];
    if ($status === 'active') {
        $sql = "UPDATE stream_sessions SET status = :status, started_at = datetime('now') WHERE session_id = :session_id";
    } elseif ($status === 'ended' || $status === 'error') {
        $sql = "UPDATE stream_sessions SET status = :status, ended_at = datetime('now') WHERE session_id = :session_id";
    } else {
        $sql = "UPDATE stream_sessions SET status = :status WHERE session_id = :session_id";
    }
    
    $stmt = $db->prepare($sql);
    return $stmt->execute([
        ':session_id' => $sessionId,
        ':status' => $status
    ]);
}

// الحصول على جلسة بث
function getStreamSession($sessionId) {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM stream_sessions WHERE session_id = :session_id");
    $stmt->execute([':session_id' => $sessionId]);
    return $stmt->fetch();
}

// إضافة رسالة signaling
function addSignalingMessage($sessionId, $fromType, $toType, $messageType, $payload) {
    $db = getDB();
    $stmt = $db->prepare("
        INSERT INTO signaling_messages (session_id, from_type, to_type, message_type, payload)
        VALUES (:session_id, :from_type, :to_type, :message_type, :payload)
    ");
    
    return $stmt->execute([
        ':session_id' => $sessionId,
        ':from_type' => $fromType,
        ':to_type' => $toType,
        ':message_type' => $messageType,
        ':payload' => $payload
    ]);
}

// الحصول على رسائل signaling غير مقروءة
function getPendingSignalingMessages($sessionId, $forType) {
    $db = getDB();
    $stmt = $db->prepare("
        SELECT * FROM signaling_messages 
        WHERE session_id = :session_id AND to_type = :to_type AND is_read = 0
        ORDER BY created_at ASC
    ");
    $stmt->execute([
        ':session_id' => $sessionId,
        ':to_type' => $forType
    ]);
    return $stmt->fetchAll();
}

// تحديد الرسائل كمقروءة
function markMessagesAsRead($messageIds) {
    if (empty($messageIds)) return true;
    
    $db = getDB();
    $placeholders = implode(',', array_fill(0, count($messageIds), '?'));
    $stmt = $db->prepare("UPDATE signaling_messages SET is_read = 1 WHERE id IN ($placeholders)");
    return $stmt->execute($messageIds);
}

// تسجيل نشاط
function logActivity($deviceId, $action, $details = '') {
    $db = getDB();
    $stmt = $db->prepare("
        INSERT INTO activity_logs (device_id, action, details, ip_address)
        VALUES (:device_id, :action, :details, :ip_address)
    ");
    
    return $stmt->execute([
        ':device_id' => $deviceId,
        ':action' => $action,
        ':details' => $details,
        ':ip_address' => getClientIP()
    ]);
}

// الحصول على سجلات النشاط
function getActivityLogs($deviceId = null, $limit = 50) {
    $db = getDB();
    
    if ($deviceId) {
        $stmt = $db->prepare("SELECT * FROM activity_logs WHERE device_id = :device_id ORDER BY created_at DESC LIMIT :limit");
        $stmt->bindValue(':device_id', $deviceId, PDO::PARAM_STR);
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    } else {
        $stmt = $db->prepare("SELECT * FROM activity_logs ORDER BY created_at DESC LIMIT :limit");
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    return $stmt->fetchAll();
}

// حذف الرسائل القديمة (أكثر من ساعة)
function cleanupOldMessages() {
    $db = getDB();
    $stmt = $db->prepare("DELETE FROM signaling_messages WHERE created_at < datetime('now', '-1 hour')");
    return $stmt->execute();
}
