<?php
/**
 * API لإدارة جلسات البث
 * Stream Sessions API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../db.php';

$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'create_session':
            // إنشاء جلسة بث جديدة (من لوحة التحكم)
            $deviceId = $input['device_id'] ?? '';
            $streamType = $input['stream_type'] ?? '';
            $cameraFacing = $input['camera_facing'] ?? null;
            
            if (empty($deviceId) || empty($streamType)) {
                throw new Exception('Device ID and stream type are required');
            }
            
            // التحقق من أن الجهاز متصل
            $device = getDevice($deviceId);
            if (!$device || !$device['is_online']) {
                throw new Exception('الجهاز غير متصل');
            }
            
            // إلغاء أي جلسات معلقة سابقة
            $db = getDB();
            $stmt = $db->prepare("
                UPDATE stream_sessions 
                SET status = 'cancelled' 
                WHERE device_id = :device_id AND status IN ('pending', 'active')
            ");
            $stmt->execute([':device_id' => $deviceId]);
            
            // إنشاء جلسة جديدة
            $sessionId = createStreamSession($deviceId, $streamType, $cameraFacing);
            
            logActivity($deviceId, 'stream_start', "Type: $streamType, Camera: $cameraFacing");
            
            echo json_encode([
                'success' => true,
                'session_id' => $sessionId,
                'message' => 'Session created successfully'
            ]);
            break;
            
        case 'get_session':
            // الحصول على معلومات الجلسة
            $sessionId = $input['session_id'] ?? $_GET['session_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $session = getStreamSession($sessionId);
            
            if (!$session) {
                throw new Exception('Session not found');
            }
            
            echo json_encode([
                'success' => true,
                'session' => $session
            ]);
            break;
            
        case 'update_status':
            // تحديث حالة الجلسة (يستخدمه التطبيق)
            $sessionId = $input['session_id'] ?? '';
            $status = $input['status'] ?? '';
            
            if (empty($sessionId) || empty($status)) {
                throw new Exception('Session ID and status are required');
            }
            
            updateStreamSession($sessionId, $status);
            
            echo json_encode([
                'success' => true,
                'message' => 'Status updated'
            ]);
            break;
            
        case 'end_session':
            // إنهاء جلسة البث
            $sessionId = $input['session_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $session = getStreamSession($sessionId);
            if ($session) {
                updateStreamSession($sessionId, 'ended');
                logActivity($session['device_id'], 'stream_end', "Session: $sessionId");
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Session ended'
            ]);
            break;
            
        case 'accept_session':
            // قبول جلسة البث (يستخدمه التطبيق)
            $sessionId = $input['session_id'] ?? '';
            $deviceId = $input['device_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $session = getStreamSession($sessionId);
            
            if (!$session) {
                throw new Exception('Session not found');
            }
            
            if ($session['device_id'] !== $deviceId) {
                throw new Exception('Session does not belong to this device');
            }
            
            updateStreamSession($sessionId, 'active');
            
            echo json_encode([
                'success' => true,
                'session' => $session
            ]);
            break;
            
        case 'reject_session':
            // رفض جلسة البث (يستخدمه التطبيق - مثلاً الشاشة مغلقة)
            $sessionId = $input['session_id'] ?? '';
            $reason = $input['reason'] ?? 'rejected';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            updateStreamSession($sessionId, 'error');
            
            // إرسال رسالة للوحة التحكم
            if ($reason === 'screen_off') {
                addSignalingMessage($sessionId, 'device', 'panel', 'screen_off', json_encode([
                    'message' => 'شاشة الهاتف مغلقة'
                ]));
            } else {
                addSignalingMessage($sessionId, 'device', 'panel', 'error', json_encode([
                    'message' => $reason
                ]));
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'Session rejected'
            ]);
            break;
            
        case 'active_sessions':
            // الحصول على الجلسات النشطة
            $db = getDB();
            $stmt = $db->query("
                SELECT s.*, d.device_name 
                FROM stream_sessions s
                LEFT JOIN devices d ON s.device_id = d.device_id
                WHERE s.status IN ('pending', 'active')
                ORDER BY s.created_at DESC
            ");
            $sessions = $stmt->fetchAll();
            
            echo json_encode([
                'success' => true,
                'sessions' => $sessions
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
