<?php
/**
 * API لـ Signaling - تبادل رسائل WebRTC
 * WebRTC Signaling API
 * 
 * هذا الملف يعمل كوسيط بين لوحة التحكم والتطبيق
 * باستخدام Long Polling بدلاً من WebSocket للبساطة
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../db.php';

$action = $_GET['action'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? $action;
}

try {
    switch ($action) {
        case 'send':
            // إرسال رسالة signaling
            $sessionId = $input['session_id'] ?? '';
            $messageType = $input['message_type'] ?? '';
            $payload = $input['payload'] ?? '';
            $fromType = $input['from_type'] ?? ($_GET['from_type'] ?? 'panel');
            
            if (empty($sessionId) || empty($messageType)) {
                throw new Exception('Session ID and message type are required');
            }
            
            // تحديد المستلم
            $toType = ($fromType === 'panel') ? 'device' : 'panel';
            
            addSignalingMessage($sessionId, $fromType, $toType, $messageType, $payload);
            
            echo json_encode([
                'success' => true,
                'message' => 'Message sent'
            ]);
            break;
            
        case 'receive':
            // استلام الرسائل المعلقة (Long Polling)
            $sessionId = $_GET['session_id'] ?? '';
            $forType = $_GET['for_type'] ?? 'panel';
            $timeout = intval($_GET['timeout'] ?? 10); // الانتظار لمدة 10 ثواني كحد أقصى
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $startTime = time();
            $messages = [];
            
            // Long Polling - الانتظار حتى وصول رسائل أو انتهاء المهلة
            while (time() - $startTime < $timeout) {
                $messages = getPendingSignalingMessages($sessionId, $forType);
                
                if (!empty($messages)) {
                    // تحديد الرسائل كمقروءة
                    $messageIds = array_column($messages, 'id');
                    markMessagesAsRead($messageIds);
                    break;
                }
                
                // الانتظار قليلاً قبل المحاولة مرة أخرى
                usleep(500000); // 0.5 ثانية
            }
            
            echo json_encode([
                'success' => true,
                'messages' => $messages
            ]);
            break;
            
        case 'poll':
            // Polling بسيط بدون انتظار
            $sessionId = $_GET['session_id'] ?? '';
            $forType = $_GET['for_type'] ?? 'device';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $messages = getPendingSignalingMessages($sessionId, $forType);
            
            if (!empty($messages)) {
                $messageIds = array_column($messages, 'id');
                markMessagesAsRead($messageIds);
            }
            
            echo json_encode([
                'success' => true,
                'messages' => $messages
            ]);
            break;
            
        case 'offer':
            // إرسال offer من لوحة التحكم
            $sessionId = $input['session_id'] ?? '';
            $sdp = $input['sdp'] ?? '';
            
            if (empty($sessionId) || empty($sdp)) {
                throw new Exception('Session ID and SDP are required');
            }
            
            addSignalingMessage($sessionId, 'panel', 'device', 'offer', json_encode([
                'type' => 'offer',
                'sdp' => $sdp
            ]));
            
            echo json_encode([
                'success' => true,
                'message' => 'Offer sent'
            ]);
            break;
            
        case 'answer':
            // إرسال answer من التطبيق
            $sessionId = $input['session_id'] ?? '';
            $sdp = $input['sdp'] ?? '';
            
            if (empty($sessionId) || empty($sdp)) {
                throw new Exception('Session ID and SDP are required');
            }
            
            addSignalingMessage($sessionId, 'device', 'panel', 'answer', json_encode([
                'type' => 'answer',
                'sdp' => $sdp
            ]));
            
            echo json_encode([
                'success' => true,
                'message' => 'Answer sent'
            ]);
            break;
            
        case 'ice_candidate':
            // إرسال ICE candidate
            $sessionId = $input['session_id'] ?? '';
            $candidate = $input['candidate'] ?? '';
            $fromType = $input['from_type'] ?? 'panel';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $toType = ($fromType === 'panel') ? 'device' : 'panel';
            
            addSignalingMessage($sessionId, $fromType, $toType, 'ice_candidate', json_encode($candidate));
            
            echo json_encode([
                'success' => true,
                'message' => 'ICE candidate sent'
            ]);
            break;
            
        case 'check_session':
            // التحقق من حالة الجلسة
            $sessionId = $_GET['session_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $session = getStreamSession($sessionId);
            
            if (!$session) {
                throw new Exception('Session not found');
            }
            
            echo json_encode([
                'success' => true,
                'session' => [
                    'session_id' => $session['session_id'],
                    'status' => $session['status'],
                    'stream_type' => $session['stream_type'],
                    'camera_facing' => $session['camera_facing']
                ]
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
