<?php
/**
 * API لإدارة الأجهزة
 * Devices Management API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../db.php';

$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'register':
            // تسجيل جهاز جديد (يستخدمه التطبيق)
            $input = json_decode(file_get_contents('php://input'), true);
            
            $deviceId = $input['device_id'] ?? '';
            $deviceName = $input['device_name'] ?? '';
            $deviceModel = $input['device_model'] ?? '';
            $androidVersion = $input['android_version'] ?? '';
            
            if (empty($deviceId)) {
                throw new Exception('Device ID is required');
            }
            
            registerDevice($deviceId, $deviceName, $deviceModel, $androidVersion);
            logActivity($deviceId, 'device_register', "Model: $deviceModel, Android: $androidVersion");
            
            echo json_encode([
                'success' => true,
                'message' => 'Device registered successfully'
            ]);
            break;
            
        case 'heartbeat':
            // نبضة قلب للتأكد من الاتصال
            $input = json_decode(file_get_contents('php://input'), true);
            $deviceId = $input['device_id'] ?? '';
            
            if (empty($deviceId)) {
                throw new Exception('Device ID is required');
            }
            
            updateDeviceStatus($deviceId, true);
            
            // التحقق من وجود طلبات بث
            $db = getDB();
            $stmt = $db->prepare("
                SELECT * FROM stream_sessions 
                WHERE device_id = :device_id AND status = 'pending'
                ORDER BY created_at DESC LIMIT 1
            ");
            $stmt->execute([':device_id' => $deviceId]);
            $pendingSession = $stmt->fetch();
            
            $response = [
                'success' => true,
                'timestamp' => time()
            ];
            
            if ($pendingSession) {
                $response['pending_stream'] = [
                    'session_id' => $pendingSession['session_id'],
                    'stream_type' => $pendingSession['stream_type'],
                    'camera_facing' => $pendingSession['camera_facing']
                ];
            }
            
            echo json_encode($response);
            break;
            
        case 'offline':
            // تسجيل الجهاز كغير متصل
            $input = json_decode(file_get_contents('php://input'), true);
            $deviceId = $input['device_id'] ?? '';
            
            if (!empty($deviceId)) {
                updateDeviceStatus($deviceId, false);
                logActivity($deviceId, 'device_offline', '');
            }
            
            echo json_encode(['success' => true]);
            break;
            
        case 'status':
            // الحصول على حالة الأجهزة (للوحة التحكم)
            $devices = getAllDevices();
            
            // تحديث الأجهزة غير المتصلة (أكثر من 30 ثانية)
            $db = getDB();
            $stmt = $db->prepare("
                UPDATE devices 
                SET is_online = 0 
                WHERE last_seen < datetime('now', '-30 seconds') AND is_online = 1
            ");
            $stmt->execute();
            
            // إعادة الحصول على الأجهزة بعد التحديث
            $devices = getAllDevices();
            
            echo json_encode([
                'success' => true,
                'devices' => $devices
            ]);
            break;
            
        case 'list':
            // قائمة الأجهزة
            $devices = getAllDevices();
            echo json_encode([
                'success' => true,
                'devices' => $devices
            ]);
            break;
            
        case 'get':
            // الحصول على جهاز محدد
            $deviceId = $_GET['device_id'] ?? '';
            
            if (empty($deviceId)) {
                throw new Exception('Device ID is required');
            }
            
            $device = getDevice($deviceId);
            
            if (!$device) {
                throw new Exception('Device not found');
            }
            
            echo json_encode([
                'success' => true,
                'device' => $device
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
