<?php
/**
 * نقطة دخول موحدة للتطبيق (Android App)
 * Unified API Endpoint for Android App
 * 
 * يستخدم هذا الملف من قبل التطبيق للتواصل مع لوحة التحكم
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Device-ID');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../db.php';

// الحصول على معرف الجهاز من الهيدر أو البيانات
$deviceId = $_SERVER['HTTP_X_DEVICE_ID'] ?? '';

$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? $_GET['action'] ?? '';

if (empty($deviceId) && isset($input['device_id'])) {
    $deviceId = $input['device_id'];
}

try {
    switch ($action) {
        // ==================== إدارة الجهاز ====================
        
        case 'register':
            // تسجيل الجهاز
            $deviceName = $input['device_name'] ?? '';
            $deviceModel = $input['device_model'] ?? '';
            $androidVersion = $input['android_version'] ?? '';
            
            if (empty($deviceId)) {
                throw new Exception('Device ID is required');
            }
            
            registerDevice($deviceId, $deviceName, $deviceModel, $androidVersion);
            logActivity($deviceId, 'device_register', "Model: $deviceModel, Android: $androidVersion");
            
            echo json_encode([
                'success' => true,
                'message' => 'Device registered'
            ]);
            break;
            
        case 'heartbeat':
            // نبضة قلب - تحديث حالة الاتصال والتحقق من طلبات البث
            if (empty($deviceId)) {
                throw new Exception('Device ID is required');
            }
            
            updateDeviceStatus($deviceId, true);
            
            // التحقق من وجود طلبات بث معلقة
            $db = getDB();
            $stmt = $db->prepare("
                SELECT session_id, stream_type, camera_facing 
                FROM stream_sessions 
                WHERE device_id = :device_id AND status = 'pending'
                ORDER BY created_at DESC LIMIT 1
            ");
            $stmt->execute([':device_id' => $deviceId]);
            $pendingSession = $stmt->fetch();
            
            $response = [
                'success' => true,
                'timestamp' => time(),
                'has_pending_stream' => false
            ];
            
            if ($pendingSession) {
                $response['has_pending_stream'] = true;
                $response['pending_stream'] = [
                    'session_id' => $pendingSession['session_id'],
                    'stream_type' => $pendingSession['stream_type'],
                    'camera_facing' => $pendingSession['camera_facing']
                ];
            }
            
            echo json_encode($response);
            break;
            
        case 'offline':
            // تسجيل قطع الاتصال
            if (!empty($deviceId)) {
                updateDeviceStatus($deviceId, false);
                logActivity($deviceId, 'device_offline', '');
            }
            echo json_encode(['success' => true]);
            break;
            
        // ==================== إدارة البث ====================
        
        case 'accept_stream':
            // قبول طلب البث
            $sessionId = $input['session_id'] ?? '';
            
            if (empty($sessionId) || empty($deviceId)) {
                throw new Exception('Session ID and Device ID are required');
            }
            
            $session = getStreamSession($sessionId);
            
            if (!$session) {
                throw new Exception('Session not found');
            }
            
            if ($session['device_id'] !== $deviceId) {
                throw new Exception('Session does not belong to this device');
            }
            
            updateStreamSession($sessionId, 'active');
            
            echo json_encode([
                'success' => true,
                'session' => [
                    'session_id' => $session['session_id'],
                    'stream_type' => $session['stream_type'],
                    'camera_facing' => $session['camera_facing']
                ]
            ]);
            break;
            
        case 'reject_stream':
            // رفض طلب البث
            $sessionId = $input['session_id'] ?? '';
            $reason = $input['reason'] ?? 'rejected';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            updateStreamSession($sessionId, 'error');
            
            // إرسال رسالة للوحة التحكم
            $messageType = 'error';
            $messagePayload = ['message' => $reason];
            
            if ($reason === 'screen_off') {
                $messageType = 'screen_off';
                $messagePayload = ['message' => 'شاشة الهاتف مغلقة'];
            }
            
            addSignalingMessage($sessionId, 'device', 'panel', $messageType, json_encode($messagePayload));
            
            echo json_encode([
                'success' => true,
                'message' => 'Stream rejected'
            ]);
            break;
            
        case 'end_stream':
            // إنهاء البث
            $sessionId = $input['session_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            updateStreamSession($sessionId, 'ended');
            logActivity($deviceId, 'stream_end', "Session: $sessionId");
            
            echo json_encode([
                'success' => true,
                'message' => 'Stream ended'
            ]);
            break;
            
        // ==================== Signaling ====================
        
        case 'get_signaling':
            // الحصول على رسائل signaling المعلقة
            $sessionId = $input['session_id'] ?? $_GET['session_id'] ?? '';
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            $messages = getPendingSignalingMessages($sessionId, 'device');
            
            if (!empty($messages)) {
                $messageIds = array_column($messages, 'id');
                markMessagesAsRead($messageIds);
            }
            
            echo json_encode([
                'success' => true,
                'messages' => array_map(function($msg) {
                    return [
                        'type' => $msg['message_type'],
                        'payload' => $msg['payload']
                    ];
                }, $messages)
            ]);
            break;
            
        case 'send_signaling':
            // إرسال رسالة signaling
            $sessionId = $input['session_id'] ?? '';
            $messageType = $input['message_type'] ?? '';
            $payload = $input['payload'] ?? '';
            
            if (empty($sessionId) || empty($messageType)) {
                throw new Exception('Session ID and message type are required');
            }
            
            addSignalingMessage($sessionId, 'device', 'panel', $messageType, 
                is_string($payload) ? $payload : json_encode($payload));
            
            echo json_encode([
                'success' => true,
                'message' => 'Signaling message sent'
            ]);
            break;
            
        case 'send_answer':
            // إرسال SDP Answer
            $sessionId = $input['session_id'] ?? '';
            $sdp = $input['sdp'] ?? '';
            
            if (empty($sessionId) || empty($sdp)) {
                throw new Exception('Session ID and SDP are required');
            }
            
            addSignalingMessage($sessionId, 'device', 'panel', 'answer', json_encode([
                'type' => 'answer',
                'sdp' => $sdp
            ]));
            
            echo json_encode([
                'success' => true,
                'message' => 'Answer sent'
            ]);
            break;
            
        case 'send_ice':
            // إرسال ICE Candidate
            $sessionId = $input['session_id'] ?? '';
            $candidate = $input['candidate'] ?? null;
            
            if (empty($sessionId)) {
                throw new Exception('Session ID is required');
            }
            
            if ($candidate) {
                addSignalingMessage($sessionId, 'device', 'panel', 'ice_candidate', 
                    is_string($candidate) ? $candidate : json_encode($candidate));
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'ICE candidate sent'
            ]);
            break;
            
        // ==================== معلومات ====================
        
        case 'get_ice_servers':
            // الحصول على خوادم ICE
            echo json_encode([
                'success' => true,
                'ice_servers' => json_decode(ICE_SERVERS, true)
            ]);
            break;
            
        case 'ping':
            // اختبار الاتصال
            echo json_encode([
                'success' => true,
                'pong' => true,
                'timestamp' => time()
            ]);
            break;
            
        default:
            throw new Exception('Invalid action: ' . $action);
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
